<?php
if (!defined('ABSPATH')) {
    exit;
}

class GFDT_Error_Logger {
    const LOG_DIR = __DIR__ . '/../logs/';
    const LOG_FILE = self::LOG_DIR . 'error.log';

    public static function init() {
        // Ensure the log directory exists
        if (!file_exists(self::LOG_DIR)) {
            mkdir(self::LOG_DIR, 0755, true);
        }

        // Set custom error handler
        set_error_handler([__CLASS__, 'custom_error_handler']);
        register_shutdown_function([__CLASS__, 'shutdown_handler']);
    }

    public static function custom_error_handler($errno, $errstr, $errfile, $errline) {
        if (strpos($errfile, plugin_dir_path(__FILE__)) === false) {
            return false;
        }
        $error_types = [
            E_ERROR             => 'Fatal Error',
            E_WARNING           => 'Warning',
            E_PARSE             => 'Parsing Error',
            E_NOTICE            => 'Notice',
            E_CORE_ERROR        => 'Core Error',
            E_CORE_WARNING      => 'Core Warning',
            E_COMPILE_ERROR     => 'Compile Error',
            E_COMPILE_WARNING   => 'Compile Warning',
            E_USER_ERROR        => 'User Error',
            E_USER_WARNING      => 'User Warning',
            E_USER_NOTICE       => 'User Notice',
            E_STRICT            => 'Runtime Notice',
            E_RECOVERABLE_ERROR => 'Catchable Fatal Error',
            E_DEPRECATED        => 'Deprecated',
            E_USER_DEPRECATED   => 'User Deprecated'
        ];

        $error_type = isset($error_types[$errno]) ? $error_types[$errno] : 'Unknown Error';

        $log_message = "[" . date("Y-m-d H:i:s") . "] [$error_type] $errstr in $errfile on line $errline\n";
        
        error_log($log_message, 3, self::LOG_FILE);
        
        return true; 
    }

    public static function shutdown_handler() {
        $error = error_get_last();
        if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
            if (strpos($error['file'], plugin_dir_path(__FILE__)) === false) {
                return; 
            }

            $log_message = "[" . date("Y-m-d H:i:s") . "] [Fatal Error] {$error['message']} in {$error['file']} on line {$error['line']}\n";
            error_log($log_message, 3, self::LOG_FILE);
        }
    }
}

// Initialize error logging
GFDT_Error_Logger::init();
